<?php

/*This file is part of <PubML>™ Publishing Tools ©2014 by Spot Grafix, Inc.

<PubML> - Web-Based eBooks

Copyright 2014 by Spot Grafix, Inc.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

<PubML>™ eBooks rely on third-party libraries offered under various open 
source licenses (MIT, Apache, etc.). The licensing terms for these files 
are described within these files. 

This <PubML>™ eBook includes files and images released under the GNU 
General Public License that are used to display YOUR content. Most of 
these files are named with a "pubml_" prefix. Any modifications you make 
to the <PubML>™ eBook platform or its css styles are considered derivative 
works that others may use and modify as provided for by, and subject to, 
the GNU General Public License. 

And any modifications or additions to such CONTENT remains the property 
of the applicable  author or rights holder and is not subject to the GNU 
General Public License. The goal is to ensure that the <PubML>™ eBook 
platform remains open source and free for anyone to modify or customize, 
while not infringing on the rights of authors to retain control, over the
rights to their content. 

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA*/

/**
 * Allows PubML EBook Publisher plugins to use their own update API.
 *
 * @author Dave Bricker
 * @version 1.0
 */
	class PUBML_Ebook_Plugin_Updater {
		private $api_url  = '';
		private $api_data = array();
		private $name     = '';
		private $slug     = '';

		/**
		 * Class constructor.
		 *
		 * @uses plugin_basename()
		 * @uses hook()
		 *
		 * @param string $_api_url The URL pointing to the custom API endpoint.
		 * @param string $_plugin_file Path to the plugin file.
		 * @param array $_api_data Optional data to send with API calls.
		 * @return void
		 */
		function __construct( $_api_url, $_plugin_file, $_api_data = null ) {
			$this->api_url  = trailingslashit( $_api_url );
			$this->api_data = urlencode_deep( $_api_data );
			$this->name     = plugin_basename( $_plugin_file );
			$this->slug     = basename( $_plugin_file, '.php');
			$this->version  = $_api_data['version'];

			// Set up hooks.
			$this->hook();
		}

		/**
		 * Set up Wordpress filters to hook into WP's update process.
		 *
		 * @uses add_filter()
		 *
		 * @return void
		 */
		private function hook() {
			add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'pre_set_site_transient_update_plugins_filter' ) );
			add_filter( 'plugins_api', array( $this, 'plugins_api_filter' ), 10, 3 );
			add_filter( 'http_request_args', array( $this, 'http_request_args' ), 10, 2 );
		}

		/**
		 * Check for Updates at the defined API endpoint and modify the update array.
		 *
		 * This function dives into the update api just when Wordpress creates its update array,
		 * then adds a custom API call and injects the custom plugin data retrieved from the API.
		 * It is reassembled from parts of the native Wordpress plugin update code.
		 * See wp-includes/update.php line 121 for the original wp_update_plugins() function.
		 *
		 * @uses api_request()
		 *
		 * @param array $_transient_data Update array build by Wordpress.
		 * @return array Modified update array with custom plugin data.
		 */
		function pre_set_site_transient_update_plugins_filter( $_transient_data ) {


			if( empty( $_transient_data ) ) return $_transient_data;

			$to_send = array( 'slug' => $this->slug );

			$api_response = $this->api_request( 'plugin_latest_version', $to_send );

			if( false !== $api_response && is_object( $api_response ) && isset( $api_response->new_version ) ) {
				if( version_compare( $this->version, $api_response->new_version, '<' ) )
					$_transient_data->response[$this->name] = $api_response;
			}
			return $_transient_data;
		}


		/**
		 * Updates information on the "View version x.x details" page with custom data.
		 *
		 * @uses api_request()
		 *
		 * @param mixed $_data
		 * @param string $_action
		 * @param object $_args
		 * @return object $_data
		 */
		function plugins_api_filter( $_data, $_action = '', $_args = null ) {
			if ( ( $_action != 'plugin_information' ) || !isset( $_args->slug ) || ( $_args->slug != $this->slug ) ) return $_data;

			$to_send = array( 'slug' => $this->slug );

			$api_response = $this->api_request( 'plugin_information', $to_send );
			if ( false !== $api_response ) $_data = $api_response;

			return $_data;
		}


		/**
		 * Disable SSL verification in order to prevent download update failures
		 *
		 * @param array $args
		 * @param string $url
		 * @return object $array
		 */
		function http_request_args( $args, $url ) {
			// If it is an https request and we are performing a package download, disable ssl verification
			if( strpos( $url, 'https://' ) !== false && strpos( $url, 'edd_action=package_download' ) ) {
				$args['sslverify'] = false;
			}
			return $args;
		}

		/**
		 * Calls the API and, if successfull, returns the object delivered by the API.
		 *
		 * @uses get_bloginfo()
		 * @uses wp_remote_post()
		 * @uses is_wp_error()
		 *
		 * @param string $_action The requested action.
		 * @param array $_data Parameters for the API action.
		 * @return false||object
		 */
		private function api_request( $_action, $_data ) {

			global $wp_version;

			$data = array_merge( $this->api_data, $_data );

			if( $data['slug'] != $this->slug )
				return;

			if( empty( $data['license'] ) )
				return;

			$api_params = array(
				'edd_action' 	=> 'get_version',
				'license' 		=> $data['license'],
				'name' 			=> $data['item_name'],
				'slug' 			=> $this->slug,
				'author'		=> $data['author'],
				'url'           => home_url()
			);
			$request = wp_remote_post( $this->api_url, array( 'timeout' => 15, 'sslverify' => false, 'body' => $api_params ) );

			if ( ! is_wp_error( $request ) ):
				$request = json_decode( wp_remote_retrieve_body( $request ) );
				if( $request && isset( $request->sections ) )
					$request->sections = maybe_unserialize( $request->sections );
				return $request;
			else:
				return false;
			endif;
		}
	}